/*---------------------------------------------------------------------------*\

    FILE....: TONETRAIN.CPP
    TYPE....: C++ Module
    AUTHOR..: Ben Kramer
    DATE....: 20/8/98

    Test program for tone training module.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/


#include "train.h"
#include "tonetrain.h"
#include "../../src/wavecpp.h"
#include "../../src/verbose.h"
#include "../../src/mess.h"

#include <vpbapi.h>

#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <libgen.h>

#define	N	24000
#define	AMP1	32000
#define	AMP2	2000
#define	F1	425
#define	F2	1000
#define	FS	8000

#define PLAIN_ENGLISH 	1
#define ENVIRONMENT 	2
#define CSOURCECODE 	3
#define VOXFILE 	4
#define WAVFILE 	5
#define TONEDEBUG 	6

#define TONE_TOL 25

int main(int argc, char *argv[]) {
	int		i,ecode,comoin,comoout,varN,noisy=0;
	int 		skip=1;
	int		tsecs=3;
	void		*wave;
	FILE		*f;
	TRAIN		train;
	VPB_DETECT	tone;

	mess_init();
	verbose(1);

	if (argc < 2){
		usage(argv);
		return -1;
	}

	// Setup default options
	comoin = VOXFILE;
	comoout = PLAIN_ENGLISH;

	// parse arguments
//	printf("Parse args\n");
	for (i=1;i<(argc-1);i++){

		if (strcmp(argv[i],"-s")==0){
			skip = atoi(argv[i+1]);
			i++;
		}
		else if (strcmp(argv[i],"-t")==0){
			tsecs = atoi(argv[i+1]);
			i++;
		}
		else if (strcmp(argv[i],"-v")==0){
			comoin = VOXFILE;
		}
		else if (strcmp(argv[i],"-w")==0){
			comoin = WAVFILE;
		}
		else if (strcmp(argv[i],"-p")==0){
			comoout = PLAIN_ENGLISH;
		}
		else if (strcmp(argv[i],"-e")==0){
			comoout = ENVIRONMENT;
		}
		else if (strcmp(argv[i],"-c")==0){
			comoout = CSOURCECODE;
		}
		else if (strcmp(argv[i],"-V")==0){
			noisy=1;
		}
		else {
			usage(argv);
			return -1;
		}
	}


	// Check buffer size
	/*
	if (tsecs == -1){
		varN=N;
	}
	else {
		varN=tsecs*FS;
	}
	*/
	varN=tsecs*FS;
	short		buf[varN];
	// fill buffer with 0's
	for(i=0; i<N; i++)
		buf[i] = 0;


	/*
	comoin = WAVFILE;
	comoout = PLAIN_ENGLISH;
	printf("Finished parse args\n");
	printf("input => %d\n",comoin);
	printf("output => %d\n",comoout);
	*/

	// process the file
	switch (comoin){
		case VOXFILE:
			{
			if(noisy){
				printf("Processing VOXFILE: %s\n",argv[argc-1]);
			}
			// open it, skipping the first couple of seconds
			// as that might contain junk like clicks etc
			f = fopen(argv[argc-1],"rt");
			fseek(f, (FS*skip)*sizeof(short), SEEK_SET);
			assert(f != NULL);
			fread(buf,sizeof(short),varN,f);
			fclose(f);

			train.OnThresh = -20;
			train.OffThresh = -40;
			train.CadenceWindow = 100;
			train.Bandwidth = 100;
			train.eThresh = (float)0.9;
			train.MagTol = 10;
			ecode = train_train(&tone, &train, buf , varN);
			}
		break;

		case WAVFILE:
			if (noisy){
				printf("Processing WAVFILE: %s\n",argv[argc-1]);
			}
			unsigned short	mode;
			unsigned long sizebytes,sizeshorts;
			int ret;
			
			// open wave file and check if it is a linear file
			wave_open_read(&wave, argv[argc-1]);
			wave_get_mode(wave, &mode);
			if (mode != VPB_LINEAR) {
				vpb_wave_close_read(wave);
				ecode = TRAIN_WAVE_FILE_NOT_LINEAR;
				break;
			}
			vpb_wave_close_read(wave);

			// open it, skipping the first couple of seconds
			// as that might contain junk like clicks etc
			f = fopen(argv[argc-1],"rt");
			fseek(f, (FS*skip)*sizeof(short), SEEK_SET);
			assert(f != NULL);
			fread(buf,sizeof(short),varN,f);
			fclose(f);

			train.OnThresh = -20;
			train.OffThresh = -40;
			train.CadenceWindow = 100;
			train.Bandwidth = 100;
			train.eThresh = (float)0.9;
			train.MagTol = 10;

			ecode = train_train(&tone,&train, buf, varN);
		break;

		default:
		break;
	}

	if (noisy){
		printf("The sample had %d cadence states...\n",tone.nstates);
		for (i=0;i<tone.nstates;i++){
			printf("\tCadence %d at %d ms \n",i,train.CadenceM[i]);
		}
		printf("Found %d tones...\n",tone.ntones);
	}
	// print the results
	if (ecode == TRAIN_OK){
		switch (comoout){
			case PLAIN_ENGLISH:
				print_plaineng(&tone);
			break;

			case ENVIRONMENT:
				print_env(&tone);
			break;

			case CSOURCECODE:
				print_csource(&tone);
			break;

			case TONEDEBUG:
				print_tonedebug(&tone);
			break;

			default:
				printf("Bugger, how did you want this info?\n");
			break;

		}

	}
	else {
		printf("An error has occured!!\n");
		print_error(ecode);
	}


	return 0;
}

void usage(char *argv[]){
	printf("usage:\t%s [<option(s)>] [<filename>]\n",basename(argv[0]));
	printf("options:\n");
	printf("\t Input options\n");
	printf("\t\t-v use vox format file <filename> for input\n");
	printf("\t\t-w use wav format file <filename> for input\n");
	printf("\t\t-s <n> Skip the first n seconds of the sample \n");
	printf("\t\t-t <n> Train using n seconds of the sample\n");
	printf("\t\t(The last 2 options are good for when you do not\n");
	printf("\t\t a clean sample of one tone. Adjust these so that\n");
	printf("\t\t tonetrain trains on only 1 tone.)\n\n");
	printf("\t Output options\n");
	printf("\t\t-p Plain English (default)\n");
	printf("\t\t-e Environment variable format\n");
	printf("\t\t-c C source code snippet\n");
/*
some suggested output options for tone detector param format:
- just the tone params in plain english
- env variable, e.g. VPB_TONE=yada yada
- C source code to implement tone detector (see tonedebug.cpp)
*/

}

void print_plaineng(VPB_DETECT *tone){
	int i;
	printf("Plain english format result:\n");
	printf("Number of Cadence states:\t %d\n",tone->nstates);
	printf("Unique ID number for this tone:\t %d\n",tone->tone_id);
	printf("Number of tones (1 or 2):\t %d\n",tone->ntones);
	printf("Frequency of first tone(Hz):\t %d\n",tone->freq1);
	printf("Bandwidth of first tone(Hz):\t %d\n",tone->bandwidth1);
	printf("Frequency of second tone(Hz):\t %d\n",tone->freq2);
	printf("Bandwidth of second tone(Hz):\t %d\n",tone->bandwidth2);
	printf("Minimum amplitude of first tone (ref 0dbm0):\t %d\n",tone->minlevel1);
	printf("Minimum amplitude of second tone (ref 0dbm0):\t %d\n",tone->minlevel2);
	printf("Allowable difference in tone powers:\t %d\n",tone->twist);
	printf("Minimum signal to noise ratio to accept tone:\t %d\n",tone->snr);
	printf("Short transitision of glitch ignored (in ms):\t %d\n",tone->glitch);
	printf("\n");
	for (i=0;i<tone->nstates;i++){
		printf("Cadence state %d\n",i);
		printf("\tType (VPB_TIMER, VPB_RISING, or VPB_FALLING): %d \n",tone->stran[i].type);
		printf("\tStran, timer mode only: %d \n",tone->stran[i].tfire);
		printf("\tMinimum tone on/off time (non timer) in ms: %d\n",tone->stran[i].tmin);
		printf("\tMaximum tone on/off time (non timer) in ms: %d\n",tone->stran[i].tmax);
		printf("\n");
	}
}

void print_env(VPB_DETECT *tone){
	int i,ontime,offtime;
	printf("Environment variable format result:\n");

	// We need to determine the Type of tone
	// from the information in the tone structure
	
	// Check if we have a continous or pulsed tone
	// First Check the number of Tones
	if (tone->ntones <= 1){
		// We have a C, P, or P2
		if (tone->nstates == 3){
			// We have a P2
			/*
			ontime = (int)(100.0*(float)tone->stran[1].tmax/(100.0 - TONE_TOL)) - (int)(100.0*(float)tone->stran[1].tmin/(100.0 + TONE_TOL));
			offtime = (int)(100.0*tone->stran[2].tmax/(100.0 - TONE_TOL)) - (int)(100.0*tone->stran[2].tmin/(100.0 + TONE_TOL));
			*/
			ontime = (int)((tone->stran[1].tmax + tone->stran[1].tmin)/2);
			//offtime = (int)((tone->stran[2].tmax + tone->stran[2].tmin)/2);
			offtime = (int)(tone->stran[2].tmax);
			//Need to work out whyt the off time is not right!
//			printf("export VPB_TONE=5,P2 ,%d ,%d ,%d ,%d\n",tone->freq1,tone->bandwidth1,ontime,offtime);
			printf("export VPB_TONE=5,P,%d,%d,%d\n",tone->freq1,tone->bandwidth1,ontime);

		}
		else if ((tone->stran[1].tmin !=0)||(tone->stran[1].tmax !=0)){
			// We have a P
			printf("export VPB_TONE=5,P,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire);
		}
		else {
			// We have a C
			printf("export VPB_TONE=5,C,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire);
		}
	}
	else {
		// We have a P3,P4,P5 etc
		if ((tone->stran[1].tfire !=0)&&(tone->stran[1].tmin ==0 )&&(tone->stran[1].tmax ==0 )&&(tone->twist ==20)){
			// We have a P3
			printf("export VPB_TONE=5,P3,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire,tone->freq2,tone->bandwidth2,tone->snr);
		}
		else if ((tone->stran[1].tfire !=0)&&(tone->stran[1].tmin ==0 )&&(tone->stran[1].tmax ==0 )&&(tone->twist !=20)){
			// We have a P6
			printf("export VPB_TONE=5,P6,%d,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire,tone->freq2,tone->bandwidth2,tone->snr,tone->twist);
		}
		else if((tone->stran[1].tfire == 0)&&((tone->stran[1].tmin != 0)||(tone->stran[1].tmax != 0))){
			// We have a P4
			ontime =   (int)(100.0*tone->stran[1].tmax/(100.0 - TONE_TOL)) - (int)(100.0*tone->stran[1].tmin/(100.0 + TONE_TOL));
			printf("export VPB_TONE=5,P4,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,ontime,tone->freq2,tone->bandwidth2,tone->snr);
		}
		else if((tone->stran[1].tfire == 0)&&((tone->stran[1].tmin != 0)||(tone->stran[1].tmax != 0)&&((tone->stran[2].tmin !=0)||(tone->stran[2].tmax != 0)))){
			// We have a P5
			ontime = (int)(100.0*tone->stran[1].tmax/(100.0 + TONE_TOL)) - (int)(100.0*tone->stran[1].tmin/(100.0 - TONE_TOL));
			offtime = (int)(100.0*tone->stran[2].tmax/(100.0 + TONE_TOL)) - (int)(100.0*tone->stran[2].tmin/(100.0 - TONE_TOL));
			printf("export VPB_TONE=5,P5,%d,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,ontime,offtime,tone->freq2,tone->bandwidth2,tone->snr);
		}
		else if((tone->twist !=20)&&(tone->stran[1].tfire == 0)&&((tone->stran[1].tmin != 0)||(tone->stran[1].tmax != 0))){
			// We have a P7
			ontime = (int)(100.0*tone->stran[1].tmax/(100.0 + TONE_TOL)) - (int)(100.0*tone->stran[1].tmin/(100.0 - TONE_TOL));
			printf("export VPB_TONE=5,P7,%d,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,ontime,tone->freq2,tone->bandwidth2,tone->snr,tone->twist);
		}
		else if((tone->twist !=20)&&(tone->stran[1].tfire == 0)&&((tone->stran[1].tmin != 0)||(tone->stran[1].tmax != 0)&&((tone->stran[2].tmin !=0)||(tone->stran[2].tmax != 0)))){
			// We have a P8
			ontime = (int)(100.0*tone->stran[1].tmax/(100.0 + TONE_TOL)) - (int)(100.0*tone->stran[1].tmin/(100.0 - TONE_TOL));
			offtime = (int)(100.0*tone->stran[2].tmax/(100.0 + TONE_TOL)) - (int)(100.0*tone->stran[2].tmin/(100.0 - TONE_TOL));
			printf("export VPB_TONE=5,P5,%d,%d,%d,%d,%d,%d,%d,%d\n",tone->freq1,tone->bandwidth1,ontime,offtime,tone->freq2,tone->bandwidth2,tone->snr,tone->twist);
		}
		else {
			printf("I have no idea how to represent that tone for use as an Environment Variable !!\n");
		}

	}
	

}



void print_tonedebug(VPB_DETECT *tone)
{
		if ((tone->stran[1].tmin !=0)||(tone->stran[1].tmax !=0)){
			// We have a P
//printf("export VPB_TONE=5,P,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire);
			printf("tonedebug -p %d %d %d ", tone->freq1, tone->bandwidth1, tone->stran[1].tfire);
			printf("\n");
		}
		else {
			// We have a C
//printf("export VPB_TONE=5,C,%d,%d,%d\n",tone->freq1,tone->bandwidth1,tone->stran[1].tfire);
			printf("tonedebug -c %d %d %d ", tone->freq1, tone->bandwidth1, tone->stran[1].tfire);
			printf("\n");
		}
};

void print_csource(VPB_DETECT *tone){
	int i;
	printf("C source code format result:\n");
	printf("static VPB_DETECT user_tone = {\n");
	printf("\t%d,\t// number of cadence states\n",tone->nstates);
	printf("\tUSER_TONE,\t// tone id ******** RENAME THIS, eg VPB_DIAL or VPB_BUSY etc *********\n");
	printf("\t%d,\t// number of tones\n",tone->ntones);
	printf("\t%d,\t// freq1\n",tone->freq1);
	printf("\t%d,\t// bandwidth1\n",tone->bandwidth1);
	printf("\t%d,\t// freq2\n",tone->freq2);
	printf("\t%d,\t// bandwidth2\n",tone->bandwidth1);
	printf("\t%d,\t// level1 \n",tone->minlevel1);
	printf("\t%d,\t// level2\n",tone->minlevel2);
	printf("\t%d,\t// twist\n",tone->twist);
	printf("\t%d,\t// SNR\n",tone->snr);
	printf("\t%d,\t// glitch duration in ms\n",tone->glitch);
	printf("\t{\n");
	for (i=0;i<tone->nstates;i++){
		printf("\t\t// Cadence %d\n",i);
		printf("\t\t{%d,\t// VPB_TIMER, VPB_RISING, or VPB_FALLING \n",tone->stran[i].type);
		printf("\t\t%d,\t// timer mode only \n",tone->stran[i].tfire);
		printf("\t\t%d,\t// minimum tone on/off time (non timer) in ms\n",tone->stran[i].tmin);
		printf("\t\t%d},\t// maximum tone on/off time (non timer) in ms\n",tone->stran[i].tmax);
	}
	printf("\t}\n");
	printf("};\n");
}

void print_error(int ecode){

	switch (ecode){
		case TRAIN_OK:
			printf("All ok!\n");
		break;
		case TRAIN_NOT_ENOUGH_CADENCE:
			printf("Not enough cadence!\n");
		break;
		case TRAIN_INVALID_ON_THRESH:
			printf("Invalid on-threshold.\n");
		break;
		case TRAIN_INVALID_OFF_THRESH:
			printf("Invalid off-threshold.\n");
		break;
		case TRAIN_INVALID_CADENCE_WINDOW:
			printf("Invalid Cadence window\n");
		break;
		case TRAIN_INVALID_BANDWIDTH	:
			printf("Invalid bandwith\n");
		break;
		case TRAIN_INVALID_E_THRESH:
			printf("Invalid E threshold\n");
		break;
		case TRAIN_WAVE_FILE_ERROR:
			printf("Wave file error!\n");
		break;
		case TRAIN_WAVE_FILE_NOT_LINEAR:
			printf("Wave file not linear!\n");
		break;
		case TRAIN_CANT_ALLOC_BUF	:
			printf("Cant allocate memeory for the buffer.\n");
		break;
		case TRAIN_WAVE_READ_ERROR:
			printf("Wave read error.\n");
		break;
		case TRAIN_NSAM_TO_SMALL:
			printf("Number of samples to small.\n");
		break;
		default:
			printf("Sorry cant translate error # %d\n",ecode);
		break;
	}
}
